<?php
/**
 * The ActionController class represents the controller in the
 * Model-View-Controller (MVC) design pattern. The ActionController receives
 * and processes all requests that change the state of the application.
 *
 * Generally, a "Model2" application is architected as follows:
 * <ul>
 * <li>The user interface will generally be created with either PHP pages,
 * which will not themselves contain any business logic. These pages represent
 * the "view" component of an MVC architecture.</li>
 * <li>Forms and hyperlinks in the user interface that require business logic
 * to be executed will be submitted to a request URI that is mapped to the
 * ActionController. The ActionController receives and processes all requests
 * that change the state of a user's interaction with the application. This
 * component represents the "controller" component of an MVC architecture.</li>
 * <li>The ActionController will select and invoke an Action class to perform
 * the requested business logic.</li>
 * <li>The Action classes will manipulate the state of the application's
 * interaction with the user, typically by creating or modifying classes that
 * are stored as session attributes. Such classes represent the "model"
 * component of an MVC architecture.</li>
 * <li>Instead of producing the next page of the user interface directly,
 * Action classes will forward control to an appropriate PHP page to produce
 * the next page of the user interface.</li>
 * </ul>
 * The standard version of ActionController implements the following logic for
 * each incoming HTTP request. You can override some or all of this
 * functionality by subclassing this class and implementing your own version of
 * the processing.
 * <ul>
 * <li>Identify, from the incoming request URI, the substring that will be used
 * to select an Action procedure.</li>
 * <li>Use this substring to map to the class name of the corresponding Action
 * class (a subclass of the Action class).</li>
 * <li>If this is the first request for a particular Action class, instantiate
 * an instance of that class and cache it for future use.</li>
 * <li>Optionally populate the properties of an ActionForm class associated
 * with this ActionMapping and cache it for future use.</li>
 * <li>Call the perform() method of this Action class. Passing in the mapping
 * and the request that were passed to the ActionController by the bootstrap.
 * </li>
 * </ul>
 *
 * The standard version of ActionController is configured based on the
 * following initialization parameters, which you will specify in the options
 * for your application. Subclasses that specialize this ActionController are
 * free to define additional initialization parameters.
 * <ul>
 * <li><b>options</b> - This sets the ActionController options.</li>
 * </ul>
 *
 * @author	Arnold Cano
 * @author	Alex Belgraver
 * @version	$Id$
 */
class ActionController extends Object
{
	/**
	 * @var	array
	 */
	var $_options;
	/**
	 * @var	ActionMappings
	 */
	var $_mappings;
	/**
	 * @var	HttpRequestServlet
	 */
	var $_request;

	/**
	 * Create a ActionController specifying the options.
	 *
	 * @access	public
	 * @param	array	$options
	 */
	function ActionController($options)
	{
		if (!is_array($options)) {
			trigger_error('Invalid options file');
			return;
		}
		$this->_options = $options;
	}
	/**
	 * Process the request.
	 *
	 * @access	public
	 * @param	array	$mappings
	 */
	function process($mappings)
	{
		if (!isset($mappings))  {
			trigger_error('Invalid mappings');
			return;
		}
		if (is_array($mappings)) $mappings =& new ActionMappings($mappings);
		$this->_mappings =& $mappings;
		$this->_createServletRequest();
		if(!defined('PHRAME_DISABLE_ERROR_HANDLING') || PHRAME_DISABLE_ERROR_HANDLING === false)
			error_reporting($this->_options[_ERROR_REPORTING]);
		//	ini_set ("display_errors","1" );
	 	//	ini_set ('error_reporting', E_ALL);
		$actionMapping = $this->_processMapping();
		$actionForm = $this->_processForm($actionMapping);
		$actionForward = $this->_processAction($actionMapping, $actionForm);
		if (is_object($actionForward)) {
			$this->_processForward($actionForward);
		}
		$this->destroy();
	}
	/**
	 * Identify and return an appropriate ActionMapping.
	 *
	 * @access	private
	 * @param	array			$mappings
	 * @param	array			$request
	 * @return	ActionMapping
	 */
	function &_processMapping()
	{
		$name = $this->_request->getAction();
		$actionMapping =&  $this->_mappings->findMapping($name);
		if (!isset($actionMapping)) {
			trigger_error("No mappings found for action '$name'");
			return;
		}
		return $actionMapping;
	}
	/**
	 * Identify and optionally return an appropriate populated ActionForm.
	 *
	 * @access	private
	 * @param	array			$mappings
	 * @return	ActionForm
	 */
	function &_processForm(&$actionMapping)
	{
		$name = $actionMapping->getName();
		//verify that a form has been mapped
		if (isset($name)) {
			$form = $this->_mappings->findForm($name);
			$type = $form[_TYPE];
			// process search-path
			if (!isset($this->_options[_FORMS_DIR])) $this->_options[_FORMS_DIR] = "forms/";
	  		$separator = ":";
			if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') $separator = ";";
			$formConfigDirs = explode($separator, $this->_options[_FORMS_DIR]);
			foreach ($formConfigDirs as $formPath) {
				if (!class_exists($type)) {
					if (file_exists ( $formPath."/$type.php" ))
						require_once $formPath."/$type.php";
				} else break;
			}
			if (!class_exists($type)) {
				trigger_error("Invalid ActionForm '$name' type '$type'");
				return;
			}
			//create and init form class
			$actionForm = new $type();
			$actionForm->reset();
			//add request parameters
			foreach ($this->_request->getParameterNames() as $name)
				$actionForm->put($name, $this->_request->getParameter($name));
			//validate the form if necesarry
			if ($actionMapping->getValidate()) {
				if (!$this->_processValidate($actionMapping, $actionForm)) {
					return;
				}
			}
			return $actionForm;
		}
		return null;
	}
	/**
	 * Call the validate() method of the specified ActionForm.
	 *
	 * @access	private
	 * @param	ActionMapping	$actionMapping
	 * @param	ActionForm		$actionForm
	 * @return	boolean
	 */
	function _processValidate(&$actionMapping, &$actionForm)
	{
		$isValid = TRUE;
		if(!defined('PHRAME_DISABLE_ERROR_HANDLING') || PHRAME_DISABLE_ERROR_HANDLING === false)
			set_error_handler($this->_options[_ERROR_HANDLER]);
		if (!$actionForm->validate()) {
			$input = $actionMapping->getInput();
			//forward errors back to view
			if(SID != "") header("Location: $input&".SID);
			if(SID == "") header("Location: $input");
			$this->destroy();
			$isValid = FALSE;
		}
		if(!defined('PHRAME_DISABLE_ERROR_HANDLING') || PHRAME_DISABLE_ERROR_HANDLING === false)
 			restore_error_handler();
		return $isValid;
	}
	/**
	 * Ask the specified Action instance to handle this request.
	 *
	 * @access	private
	 * @param	ActionMapping	$actionMapping
	 * @param	ActionForm		$actionForm
	 * @return	ActionForward
	 */
	function &_processAction(&$actionMapping, &$actionForm)
	{
		$type = $actionMapping->getType();
		$name = $type;
		if ($pos = strrpos($type,"/")) // strip directory from $type here.
			$name = substr($type,++$pos,strlen($type)-$pos);
		// process search-path
		if (!isset($this->_options[_ACTIONS_DIR])) $this->_options[_ACTIONS_DIR] = "actions/";
  		$separator = ":";
		if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') $separator = ";";
		$actionConfigDirs = explode($separator, $this->_options[_ACTIONS_DIR]);
		foreach ($actionConfigDirs as $actionPath) {
			if (!class_exists($type)) {
				if (file_exists ( $actionPath."/$type.php" ))
					require_once $actionPath."/$type.php";
			} else break;
		}
		if (!class_exists($type)) {
			trigger_error("Invalid Action '$name' type '$type'");
			return;
		}
		$action = new $type();
		set_error_handler($this->_options[_ERROR_HANDLER]);
		$actionForward = $action->perform($actionMapping, $actionForm);
		restore_error_handler();
		return $actionForward;
	}
	/**
	 * Forward to the specified destination.
	 *
	 * @access	private
	 * @param	ActionForward	$actionForward
	 */
	function _processForward(&$actionForward)
	{
		$redirect = $actionForward->getRedirect();
		$path = $actionForward->getPath();
		if (!$redirect) {
			if(SID != "") header("Location: $path&".SID);
			if(SID == "") header("Location: $path");
		} else {
			$_SESSION = array();
			session_destroy();
			header("Location: $path");
		}
		$this->destroy();
	}
	/**
	 * Create HttpRequestServlet object. Reuses existing object.
	 * post: the $this->_request exists correctly.
	 * @access private
	 */
	function _createServletRequest()
	{
		$this->_request =& new HttpServletRequest();
		$this->_request->setController($this);
	}
	/**
	 * Finish off the request and take down the controller.
	 * @access protected
	 * @return void
	 */
	function destroy()
	{
		// delete things that don't need saving.
		unset($this->_options);
		unset($this->_request);
		unset($this->_mappings);
		// and... end the request
		// exit;
	}
}
?>
